// -----------------------------------------------------------------------------------------------------------------------
// Constructing and rendering a coil-shaped space curve in CSharpCalc 3D
// -----------------------------------------------------------------------------------------------------------------------

// 2D surface initialization
// -----------------------------------------------------------------------------------------------------------------------
CSCDisplay.SetSurfaceSize(1000, 800);
CSCDisplay.SetPhysicalSize(-1, -1, 1, 1);
CSCDisplay.Clear(Color.Black);

// 3D scene setup

// -----------------------------------------------------------------------------------------------------------------------
// Place the camera at position (5.0, 5.0, 5.0) in global coordinates
CSC3DScene.SetCameraPosition(5.0, 5.0, 5.0);

// Let the camera point towards the origin of the global coordinate system
CSC3DScene.SetCameraLookAt(0.0, 0.0, 0.0);

// Enable directional lighting
CSC3DScene.SetDirectionalLightColor(255, 255, 255);
CSC3DScene.SetLightDirection(-0.5, -0.5, -1.5);

// Disable ambient lighting
CSC3DScene.SetAmbientLightColor(0, 0, 0);

// 3D scene construction
// -----------------------------------------------------------------------------------------------------------------------
CSC3DScene.BeginConstruction();

// create a coil-shaped psace curve represented as CSCRealVectorFunction
CSCRealVectorFunction coil = Coil.Create();

// Create a rectangular path object as object base
CSC3DObject BaseOBJ = CSC3DFactory.CreateRectangle(0.1, 0.05);

// Create a filled rectangle of the same size to be used for the caps
CSC3DObject CapOBJ = CSC3DFactory.CreatePlane(0.1, 0.05);

// Create the path object representing the coil space curve
CSC3DObjectGroup G = CSC3DFactory.CreatePathObject(BaseOBJ, CapOBJ, coil);

// Adapt the coil surface properties. Caps are highlighted in orange color.
G[0].Flatten();
G[0].SetDiffuse(100, 100, 100);
G[0].SetEmissive(50, 50, 20);
G[1].SetEmissive(250, 80, 20);
G[2].SetEmissive(250, 80, 20);

// Finally add the coil object (group) to the scene.
CSC3DScene.Add(G);

CSC3DScene.EndConstruction();

// Scene rendering and presentation
// -----------------------------------------------------------------------------------------------------------------------
CSCDisplay.BeginRendering();
CSC3DScene.Render();
CSCDisplay.EndRendering();
CSCDisplay.Present();


/* <classcode> */

// This class creates a CSCRealVectorFunction representing a coil
public static class Coil
{
 public static CSCRealVectorFunction Create()
 {
  CSCRealVectorFunction curve = new CSCRealVectorFunction();

  double phi = 10.0;
  double x = 1.0;
  double y = 0.0;
  double z = 0.0;

  double cp = Math.Cos(Math.PI*phi/180.0);
  double sp = Math.Sin(Math.PI*phi/180.0);

  double x0 = x;
  double y0 = y;
  double z0 = z;

  curve.Samples.Add(new CSCRealVector(new double[] {x, y, z}));

  for(int j = 0; j <= 144; j++)
  {
   x = cp*x0 - sp*y0;
   y = sp*x0 + cp*y0;
   z = z0 + 0.25*j*phi/360.0;

   x0 = x;
y0 = y;

   curve.Samples.Add(new CSCRealVector(new double[] {x, y, z}));
  }
  return curve;
 }
}