/*

 This script is part of the CSharpCalc documentation. Use and modify this script
 free of charge but on your own risk, without warranty and without liability. The
 sole purpose of this script is the practical documentation of certain features
 of the CSharpCalc software. It is not suited for any commercial use. Whenever
 given, additional instructions contained in this script are important must be
 followed prior to starting this script in CSharpCalc.

 Version requirement: CSharpCalc version 1.0 all revisions.
*/
// ------------------------------------------------
// Mandelbrot set simulation
// ------------------------------------------------
// Set the active colormap. Alternatives are available and documented
// in the CSCColorMap class interface.
CSCColorMap.SetTo(CSCColorMapType.RedToYellow);

// Increase the iteration depths for eventual close ups
Mandelbrot.IterationDepth = 50;

// Initialize console and display
CSCConsole.Clear();
CSCDisplay.SetSurfaceSize(600, 450);

// set the physical size to the rectangle enclosing the Mandelbrot set.
CSCDisplay.SetPhysicalSize(-2.0, -1.2, 0.5, 1.2);
CSCDisplay.Clear(Color.Black);

// render the Mandelbrot set
CSCDisplay.BeginRendering();

for(int j = 0; j < CSCDisplay.PixelHeight; j++)
{
 for(int i = 0; i < CSCDisplay.PixelWidth; i++)
 {
  // Use XFromPixel and YFromPixel to find the physical position of a pixel
  double x = CSCDisplay.XFromPixel(i);
  double y = CSCDisplay.YFromPixel(j);

  Color c = Color.Black;
  int speed = Mandelbrot.DivergenceSpeed(x, y);
  if(speed > 0) { c = CSCColorMap.GetColor(speed); }
  CSCDisplay.SetPixel(i, j, c);
 }

 // Report the rendering progress to the console every 100 lines.
 if(j > 0 && j % 100 == 0) { CSCConsole.Writeln("Lines: " + j.ToString()); }
}

CSCConsole.Writeln("Lines: " + CSCDisplay.PixelHeight.ToString());

CSCDisplay.EndRendering();
CSCDisplay.Present();
CSCDataIO.WriteSurface();
CSCConsole.Writeln("Mandelbrot finished");

/* <classcode> */
// we need a function for measuring the speed of divergence...
public static class Mandelbrot
{
 public static int DivergenceSpeed(double x, double y)
 {
  // Use the build-in complex number class CSCComplex for calculations
  CSCComplex c = new CSCComplex(x, y);
  CSCComplex z1 = new CSCComplex(0.0, 0.0);
  CSCComplex z2 = new CSCComplex(0.0, 0.0);
  for(int i = 0; i < IterationDepth; i++)
  {
   z2 = z1*z1 + c;
   z1 = z2;
   if(z1.R > 4) { return (int)(i*255.0/IterationDepth); }
  }
  return 0;
 }
 public static int IterationDepth = 1;
}