/*

 This script is part of the CSharpCalc documentation. Use and modify this script
 free of charge but on your own risk, without warranty and without liability. The
 sole purpose of this script is the practical documentation of certain features
 of the CSharpCalc software. It is not suited for any commercial use. Whenever
 given, additional instructions contained in this script are important must be
 followed prior to starting this script in CSharpCalc.

 Version requirement: CSharpCalc version 1.0 all revisions.
*/
CSCDisplay.SetSurfaceSize(800, 600);

CSCFunctionPlot.Initialize();
CSCFunctionPlot.SetMargins(10, 10, 10, 10);

// Set a custom color for the background and the text
CSCFunctionPlot.SetBackground(Color.FromArgb(255, 0, 0, 0));
CSCFunctionPlot.SetLabelColor(Color.Gainsboro);

CSCFunctionPlot.SetStyle(true, true, false);
CSCFunctionPlot.SetNumberOfGridLines(4, 3);

// specify custom colors for drawing the coordinate frame
CSCFunctionPlot.SetAxesStyle(Color.FromArgb(255, 0, 80, 140), 2.0);
CSCFunctionPlot.SetGridStyle(Color.FromArgb(255, 0, 80, 140), 1.0);
CSCFunctionPlot.SetFrameStyle(Color.FromArgb(255, 0, 80, 140), 1.0);

CSCFunctionPlot.SetLegendTextOffset(2, 10);

// Draw the parametric function in a very light gray (Gainsboro))
CSCFunctionPlot.SetStyle(Color.Gainsboro, 1.0);

// generate an instance of the parametric curve defined in the classcode
// This instance will be used again in the custom graphics block below.
FImplicit f = new FImplicit(3, 4, 1.0, 200.0, 1.0, 150, 20000, 0.0, 360.0);

// plot the parametric curve
CSCFunctionPlot.ParametricPlot(f);

// add custon graphics.
CSCDisplay.BeginRendering();
CSCRender2D.SetFillColor(Color.White);

// A custom headline
CSCRender2D.DrawCenteredText(0.0, 1.1, "A Parametric plot", 0, 0);

// Render a text line containing the parameters a, b, c and d at the bottom
CSCRender2D.DrawText(1.0, -2.15, f.ToString(), 0, 0);

CSCDisplay.EndRendering();
CSCDisplay.Present();
CSCDataIO.WriteSurface();

/* <classcode> */
// define the parametric curve as CSCRealVectorFunction
public class FImplicit : CSCRealVectorFunction
{
 public FImplicit(int j, int k,
                 double a, double b,
                 double c, double d,
                 int numsamples, double xmin, double xmax) : base(xmin, xmax)
 {
  _j = j; _k = k;
  _a = a; _b = b; _c = c; _d = d;

  // resample the curve and disable interpolation mode
  Resample(numsamples);
  SetInterpolationMode(false);
 }

 public override CSCRealVector Value(double x)
 {
  CSCRealVector v = new CSCRealVector(2);

  double t = x*Math.PI/180.0;

  v[0] = Math.Cos(_a*t) - Math.Pow(Math.Cos(_b*t), _j);
  v[1] = Math.Sin(_c*t) - Math.Pow(Math.Sin(_d*t), _k);

  return v;
 }

 // we also override ToString() and return a formatted parameter string.
 public override string ToString()
 {
  string s = "Parameters: a=" + _a.ToString("#####.##") +
                       ", b=" + _b.ToString("#####.##") +
                       ", c=" + _c.ToString("#####.##")
                     + ", d=" + _d.ToString("#####.##");

  return s;
 }

 // the parameters of the curve as private class attributes. specified in
 // the constructor
 private int _j = 1;
 private int _k = 1;
 private double _a = 1.0;
 private double _b = 1.0;
 private double _c = 1.0;
 private double _d = 1.0;
}