/*

 This script is part of the CSharpCalc documentation. Use and modify this script
 free of charge but on your own risk, without warranty and without liability. The
 sole purpose of this script is the practical documentation of certain features
 of the CSharpCalc software. It is not suited for any commercial use. Whenever
 given, additional instructions contained in this script are important must be
 followed prior to starting this script in CSharpCalc.

 Version requirement: CSharpCalc version 1.0 all revisions.
*/

/*

Rendering of an animated phase plot in CSharpCalc

WARNING! This script generates 800 animation frames in your data directory.
Make sure to configure the data directory appropriately BEFORE starting this script.

*/


/* <maincode> */
double lambda = -2.5;
double delta = 0.1;
double mu = 0.8;
double X0 = 2.0;
double P0 = 0.0;
double dt = 0.0001;
double T = 40.0;
int numframes = 800;
double DT = T/numframes;

CSCDisplay.SetSurfaceSize(600, 450);

CenteredCoordinateSystem.Initialize(2.2, 4.5, 10.0, 5.0, "x", "p");

// set the name of the files to 'phaseplot' and enable file counting
// with 1 (i.e. 00000001) as starting index
CSCDataIO.SetFilePattern("phaseplot", "");
CSCDataIO.EnableFileCounting(1);

// make a preview of the full trajectory and store it in HH.
Oscillator1D.Initialize(lambda, mu, delta, X0, P0, dt, T);
Oscillator1D.Solve();
CSCRealVectorFunction HH = new CSCRealVectorFunction(Oscillator1D.H.Serialize());

Oscillator1D.Initialize(lambda, mu, delta, X0, P0, dt, T);

CSCDisplay.Clear(Color.White);

// iterate and render the phaseplot step by step
for(int i = 1; i <= numframes; i++)
{
 CSCDisplay.BeginRendering();

 // solve the oscillator system step by step
 Oscillator1D.Solve(DT, i*DT);

 // draw the coordinate frame first...
 CenteredCoordinateSystem.Draw();

 // then draw the preview trajectory in a very light gray color..
 CSCRender2D.SetLineStyle(Color.Gainsboro, 1.6f);
 CSCRender2D.DrawTrajectory(HH, 0, 1);

 // finally draw the next piece of the animated trajectory in with
 // a darker pen.
 CSCRender2D.SetLineStyle(Color.DarkSlateGray, 1.6f);
 CSCRender2D.DrawTrajectory(Oscillator1D.H, 0, 1);

 // Write the frame. Since file counting is enabled SCSDataIO
 // does the indexing automatically
 CSCDataIO.WriteSurface();

 // give a progress report in the console for every rendered frame
 CSCConsole.Writeln("Rendered animation frame: " + i.ToString("0000") + " / " + numframes.ToString());
 CSCDisplay.EndRendering();

 // this color choice initializes the surface to be transparent
 CSCDisplay.Clear(Color.FromArgb(0, 0, 0, 0));
}

// finally report completion in the console.
CSCConsole.Writeln("Phase plot animation finished!");


/* <classcode> */
public static class Oscillator1D
{
 public static void Initialize(double lambda, double mu, double delta, double x0, double p0, double dt, double T)
 {
  _lambda = lambda;
  _mu = mu;
  _delta = delta;
  _x0 = x0; _p0 = p0;
  _dt = dt; _T = T;
  __x =  _x0;
  __p =  _p0;
  __xx = __x; __pp = __p;

  _x = new CSCRealFunction(0.0, _T);
  _p = new CSCRealFunction(0.0, _T);
  _h = new CSCRealVectorFunction();
 }

 public static void Solve()
 {
  int nsteps = (int)(_T/_dt);
  X.Append(__x); P.Append(__p);
  H.Append(new CSCRealVector(new double[] { __x, __p }));
  for(int i = 0; i < nsteps; i++)
  {
   __xx = __x + __p*_dt + 0.5*(F1(__x) - _delta*__p)*_dt*_dt;
   __pp = __p + (F1(__x) - _delta*__p)*_dt + 0.5*(F2(__x)*__p - _delta*_delta*__p - _delta*F1(__x))*_dt*_dt;
   __x = __xx; __p = __pp;
   X.Append(__x); P.Append(__p);
   H.Append(new CSCRealVector(new double[] { __x, __p }));
  }
 }
 public static void Solve(double DT, double MaxT)
 {
  int nsteps = (int)(DT/_dt);
  X.Append(__x); P.Append(__p);
  X.XMax = MaxT; P.XMax = MaxT;
  H.Append(new CSCRealVector(new double[] { __x, __p }));
  for(int i = 0; i < nsteps; i++)
  {
   __xx = __x + __p*_dt + 0.5*(F1(__x) - _delta*__p)*_dt*_dt;
   __pp = __p + (F1(__x) - _delta*__p)*_dt + 0.5*(F2(__x)*__p - _delta*_delta*__p - _delta*F1(__x))*_dt*_dt;
   __x = __xx; __p = __pp;
   X.Append(__x); P.Append(__p);
   H.Append(new CSCRealVector(new double[] { __x, __p }));
  }
 }

 public static CSCRealFunction X { get { return _x; } }
 public static CSCRealFunction P { get { return _p; } }
 public static CSCRealVectorFunction H {  get { return _h; } }

 // the potential V(x)
 public static CSCRealFunction V(double min, double max, double dx)
 {
  CSCRealFunction v = new CSCRealFunction(min, max);
  int nsteps = (int)((max - min)/dx);
  double x = min;
  double y = x*x*x*x + _lambda*x*x + _mu*x;
  v.Samples.Add(y);

  for(int i = 0; i < nsteps; i++)
  {
   x += dx;
   y = x*x*x*x + _lambda*x*x + _mu*x;
   v.Samples.Add(y);
  }
  return v;
 }

 // the negative first derivative of the potential a.k.a force
 private static double F1(double x)
 {
  return -(4*x*x*x + 2.0*_lambda*x + _mu);
 }

 // the negative second derivative of the potential
 private static double F2(double x)
 {
  return -(12*x*x + 2.0*_lambda);
 }
 private static double _lambda = 1.0;
 private static double _mu = -1.0;
 private static double _delta = 0.0;
 private static double _x0 = 1.0;
 private static double _p0 = 0.0;
 private static double _dt = 0.01;
 private static double _T = 10.0;

 private static CSCRealFunction _x = null;
 private static CSCRealFunction _p = null;
 private static CSCRealVectorFunction _h = null;

 private static double __x = 0.0;
 private static double __p = 0.0;
 private static double __xx = 0.0;
 private static double __pp = 0.0;
}

public static class CenteredCoordinateSystem
{
 public static void Initialize(double xsize, double ysize, double xgridlines, double ygridlines, string xlabel, string ylabel)
 {
  _xsize = xsize; _ysize = ysize;
  _xgridlines = xgridlines; _ygridlines = ygridlines;

  _ccs = new CSCCartesianCoordinateSystem(-_xsize, -_ysize, _xsize, _ysize, _xsize/_xgridlines, _ysize/_ygridlines);
  _ccs.XLabel = xlabel;
  _ccs.YLabel = ylabel;
  // Let the surface contain the coordinate system with a 20% margin
  CSCDisplay.SetPhysicalSize(_ccs, 0.2*_xsize, 0.2*_ysize);
 }

 public static void Draw()
 {
  // Draw the coordinate system.
  CSCRender2D.DrawGrid(_ccs, Color.Gainsboro, 1);
  CSCRender2D.DrawFrame(_ccs, Color.Black, 1);
  CSCRender2D.DrawAxes(_ccs, Color.Black, 2);

  CSCRender2D.SetLineStyle(Color.Black, 2);
  CSCRender2D.DrawXLabelAtAxis(_ccs, 0, 0);
  CSCRender2D.DrawYLabelAtAxis(_ccs, 0, 0);

  // draw the units along the x-axis
  CSCRender2D.DrawCenteredText(-_xsize, -_ysize, _xsize.ToString("-0.00"), 0, -10);
  CSCRender2D.DrawCenteredText(0.0, -_ysize, "0.00", 0, -10);
  CSCRender2D.DrawCenteredText(_xsize, -_ysize, _xsize.ToString("0.00"), 0, -10);

  // draw the units along the y-axis
  CSCRender2D.DrawCenteredText(-_xsize, -_ysize, _ysize.ToString("-0.00"), -20, 5);
  CSCRender2D.DrawCenteredText(-_xsize, 0.0, "0.00", -20, 0);
  CSCRender2D.DrawCenteredText(-_xsize, _ysize, _ysize.ToString("0.00"), -20, -5);
 }

 private static double _xsize = 1.0;
 private static double _ysize = 1.0;
 private static double _xgridlines = 5.0;
 private static double _ygridlines = 5.0;
 private static CSCCartesianCoordinateSystem _ccs = null;
}